/***************************************************************************
*
* Copyright 2010, 2011 BMW Car IT GmbH
* Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
* SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
* FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
* SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
* RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
* CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
* CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*
****************************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <linux/input.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "WindowSystems/WaylandInputEvent.h"
#include "WindowSystems/WaylandBaseWindowSystem.h"

#define HAVE_MKOSTEMP 1

/////////////////////////////////////////////////////////////////////////////

#ifndef HAVE_MKOSTEMP
static int
setCloexecOrClose(int fd)
{
    long flags;
    if (fd == -1)
        return -1;

    flags = fcntl(fd, F_GETFD);
    if (flags == -1)
        goto err;

    if (fcntl(fd, F_SETFD, flags | FD_CLOEXEC) == -1)
        goto err;

    return fd;
err:
    close(fd);
    return -1;
}
#endif

static int
createTmpFileCloexec(char *tmpname)
{
    int fd;
#ifdef HAVE_MKOSTEMP
    fd = mkostemp(tmpname, O_CLOEXEC);
    if (fd >= 0)
    {
        unlink(tmpname);
    }
#else
    fd = mkstemp(tmpname);
    if (fd >= 0)
    {
        fd = setCloexecOrClose(fd);
        unlink(tmpname);
    }
#endif
    return fd;
}

/////////////////////////////////////////////////////////////////////////////

WaylandInputEvent::WaylandInputEvent(WaylandBaseWindowSystem *windowSystem)
: m_windowSystem(windowSystem)
, m_inputDevice(NULL)
, m_wlEventSource(NULL)
, m_fd(0)
{
    strcpy(m_nameSeat, "seat0");
    initInputEvent(m_nameSeat, windowSystem->getInpManager());
}

WaylandInputEvent::WaylandInputEvent(WaylandBaseWindowSystem *windowSystem,
    char *pSeatName, InputManager *pIpManager)
: m_windowSystem(windowSystem)
, m_inputDevice(NULL)
, m_wlEventSource(NULL)
, m_fd(0)
{
    strcpy(m_nameSeat, pSeatName);
    initInputEvent(m_nameSeat, pIpManager);
}

WaylandInputEvent::~WaylandInputEvent()
{
    if (!m_wlEventSource)
        wl_event_source_remove(m_wlEventSource);
}

void
WaylandInputEvent::initInputEvent(char *pSeatName, InputManager *pInputManager)
{

    m_inputDevice = new WaylandInputDevice(m_windowSystem->getNativeDisplayHandle(), pSeatName, pInputManager);
    if (!m_inputDevice)
    {
        LOG_WARNING("WaylandInputEvent",
                    "Failed to create WaylandInputDevice");
        return;
    }
    else
    {
        LOG_DEBUG("WaylandInputEvent",
                  "Initialise input device for "
                  "native display handle="
                  << m_windowSystem->getNativeDisplayHandle() << ", "
                  "input device=" << m_inputDevice);
    }
}

void
WaylandInputEvent::setupInputEvent()
{
    LOG_DEBUG("WaylandInputEvent",
              "Setup");
}

void
WaylandInputEvent::initPointerDevice()
{
    LOG_DEBUG("WaylandInputEvent",
              "Device=" << m_inputDevice << ", "
              "has pointer=" << m_inputDevice->hasPointer());

    if (m_inputDevice->hasPointer())
        return;
    m_inputDevice->initPointerDevice();
}

void
WaylandInputEvent::initKeyboardDevice(struct xkb_keymap *keymap)
{
    LOG_DEBUG("WaylandInputEvent",
              "Device=" << m_inputDevice << ", "
              "has keyboard=" << m_inputDevice->hasKeyboard() << ", "
              "keymap=" << keymap);

    if (m_inputDevice->hasKeyboard())
        return;

    m_inputDevice->initKeyboardDevice();
}

void
WaylandInputEvent::initTouchDevice()
{
    LOG_DEBUG("WaylandInputEvent",
              "Device=" << m_inputDevice << ", "
              "has touch=" << m_inputDevice->hasTouch());

    if (m_inputDevice->hasTouch())
        return;
    m_inputDevice->initTouchDevice();
}
